import { Suspense } from "react"
import { notFound } from "next/navigation"
import { Header } from "@/components/header"
import { MovieDetails } from "@/components/movie-details"
import { MovieTrailer } from "@/components/movie-trailer"
import { WhereToWatch } from "@/components/where-to-watch"
import { SimilarMovies } from "@/components/similar-movies"
import { AdultContentWarning } from "@/components/adult-content-warning"
import { LoginRequiredOverlay } from "@/components/login-required-overlay"
import { Skeleton } from "@/components/ui/skeleton"
import { createServerClient } from "@/lib/supabase"

async function getMovie(id: string) {
  try {
    const res = await fetch(
      `https://api.themoviedb.org/3/movie/${id}?api_key=${process.env.TMDB_API_KEY}&append_to_response=videos,credits,watch/providers`,
      { next: { revalidate: 3600 } },
    )

    if (!res.ok) {
      console.error(`TMDB movie fetch failed (${res.status}) for id ${id}`)
      return null
    }

    return res.json()
  } catch (err) {
    console.error("TMDB movie fetch error:", err)
    return null
  }
}

async function checkUserAuth() {
  try {
    const supabase = createServerClient()
    const {
      data: { session },
    } = await supabase.auth.getSession()
    return !!session?.user
  } catch (error) {
    return false
  }
}

export default async function MoviePage({
  params,
}: {
  params: { id: string }
}) {
  const [movie, isLoggedIn] = await Promise.all([getMovie(params.id), checkUserAuth()])

  if (!movie) {
    notFound()
  }

  // Check if this is adult content and user is not logged in
  if (movie.adult && !isLoggedIn) {
    return (
      <div className="min-h-screen bg-background">
        <Header />
        <div className="container mx-auto px-4 py-8">
          <LoginRequiredOverlay
            title={movie.title}
            type="movie"
            description="This movie contains mature content and requires user authentication to view."
          />
        </div>
      </div>
    )
  }

  const movieContent = (
    <>
      <div className="container mx-auto px-4 py-8">
        <MovieDetails movie={movie} />

        <div className="mt-12 grid grid-cols-1 lg:grid-cols-3 gap-8">
          <div className="lg:col-span-2 space-y-8">
            <Suspense fallback={<Skeleton className="aspect-video w-full" />}>
              <MovieTrailer videos={movie.videos?.results || []} />
            </Suspense>

            <div>
              <h2 className="text-2xl font-bold mb-6">Similar Movies</h2>
              <Suspense fallback={<div>Loading similar movies...</div>}>
                <SimilarMovies movieId={params.id} />
              </Suspense>
            </div>
          </div>

          <div>
            <WhereToWatch providers={movie["watch/providers"]?.results || {}} />
          </div>
        </div>
      </div>
    </>
  )

  return (
    <div className="min-h-screen bg-background">
      <Header />

      {movie.adult && isLoggedIn ? (
        <AdultContentWarning isAdult={movie.adult} title={movie.title} type="movie">
          {movieContent}
        </AdultContentWarning>
      ) : (
        movieContent
      )}
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { id: string }
}) {
  const movie = await getMovie(params.id)

  if (!movie) {
    return {
      title: "Movie Not Found",
    }
  }

  return {
    title: `${movie.title} - MoviesWeb`,
    description: movie.overview,
    openGraph: {
      title: movie.title,
      description: movie.overview,
      images: movie.poster_path ? [`https://image.tmdb.org/t/p/w500${movie.poster_path}`] : [],
    },
  }
}
